<?php

namespace App\Http\Controllers;

use App\DTOs\SalesChannel\SalesChannelCreateDto;
use App\DTOs\SalesChannel\SalesChannelUpdateDto;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use Exception;
use App\Models\SalesChannel;
use App\Repositories\Interfaces\ISalesChannelRepository;

class SalesChannelController extends Controller
{

    private $saleChannelRepository;
    private $errorConnection = "";

    public function __construct(Request $request, ISalesChannelRepository $saleChannelRepository)
    {
        if (isset($request['business_information'])) {
            $business_information = $request['business_information'];
            $ruc = $business_information['ruc_number'];

            $existe_conexion = $this->setDBClient($ruc);

            if ($existe_conexion['error']) {
                $this->errorConnection = $existe_conexion['message'];
                return $this->messageError("Se espera un numero de RUC valido.", 422);
            } else {
                $this->saleChannelRepository = $saleChannelRepository;
            }
        } else {
            $this->errorConnection = 'Error en la conexion. Se espera numero de RUC';
        }
    }

    public function index(Request $request)
    {
        if ($this->errorConnection != '') {
            return $this->messageError($this->errorConnection, 500);
        }

        $result = $this->saleChannelRepository->paginate($request->all());

        if ($result) {
            if ($result['total'] > 0) {
                return $this->successResponse($result, "Canales de venta encontrados correctamente.", 200);
            } else {
                return $this->successResponse($result, "No hay canales de venta.", 200);
            }
        } else {
            return $this->errorResponse("No hay canales de venta.", 404);
        }
    }

    public function show(int $id)
    {
        if ($this->errorConnection != '') {
            return $this->messageError($this->errorConnection, 500);
        }

        $result = $this->saleChannelRepository->find($id);

        if ($result) {
            return $this->successResponse($result, "Canal de venta encontrada correctamente", 200);
        } else {
            return $this->errorResponse("Canal de venta no encontrada", 404);
        }
    }

    public function store(Request $request)
    {
        try {
            if ($this->errorConnection != '') {
                return $this->messageError($this->errorConnection, 500);
            }

            $this->validate($request, [
                'code' => 'required|alpha_dash|max:10',
                'name' => 'required|max:50',
                'description' => 'max:255',
                'created_by' => 'required|numeric|digits_between:1,11'
            ], [
                'code.alpha_dash' => 'El codigo del canal no puede tener caracteres especiales.',
                'code.required' => 'El codigo del canal es requerido.',
                'code.max' => 'El codigo del canal no puede superar los :max caracteres.',

                'name.required' => 'El nombre es requerido.',
                'name.max' => 'El nombre no puede superar los :max caracteres.',

                'description.max' => 'La descripcion no debe superar los :max caracteres',

                'created_by.required' => 'El campo created_by es requerido.',
                'created_by.numeric' => 'El campo created_by debe ser numerico.',
                'created_by.digits_between' => 'El campo created_by no puede superar los :max caracteres.',
            ]);
            $business_information = $request['business_information'];
            $ruc = $business_information['ruc_number'];
            //Validacion personalizado
            $code = $request['code'];
            $name = $request['name'];
            $dataMessage = [];
            //VALIDACION CODIGO Y NOMBRE
            $codeAndNameExist = SalesChannel::where('code', '=', $code)->where('name', '=', $name)->where('status', '=', 1)->get();
            if (count($codeAndNameExist) >= 1) {
                $dataMessage['code'][] = "El codigo y nombre ingresado ya existe";
                return $this->errorResponse($dataMessage, 422);
            }

            $store = new SalesChannelCreateDto($request->except(array('business_information', 'items')));
            $store->ruc = $ruc;
            $result = $this->saleChannelRepository->store($store);
            return $this->successResponse($result, "El canal de venta fue creada correctamente", 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 422);
        }
    }

    public function update(int $id, Request $request)
    {
        try {

            if ($this->errorConnection != '') {
                return $this->messageError($this->errorConnection, 500);
            }

            $result = $this->saleChannelRepository->find($id);

            if ($result) {
                $this->validate($request, [
                    'code' => 'required|alpha_dash|max:10',
                    'name' => 'required|max:50',
                    'description' => 'max:255',
                    'updated_by' => 'required|numeric|digits_between:1,11',
                ], [
                    'cod_vend.alpha_dash' => 'El codigo de Vendedor no puede tener caracteres especiales.',
                    'cod_vend.required' => 'El codigo de Vendedor es requerido.',
                    'cod_vend.max' => 'El codigo de Vendedor no puede superar los :max caracteres.',

                    'name.required' => 'El nombre es requerido.',
                    'name.max' => 'El nombre no puede superar los :max caracteres.',

                    'description.max' => 'La descripcion no debe superar los :max caracteres',

                    'updated_by.required' => 'El campo updated_by es requerido.',
                    'updated_by.numeric' => 'El campo updated_by debe ser numerico.',
                    'updated_by.digits_between' => 'El campo updated_by no puede superar los :max caracteres.',
                ]);

                //Validacion personalizado
                $code = $request['code'];
                $name = $request['name'];
                $dataMessage = [];
                //VALIDACION CODIGO Y NOMBRE
                $codeAndNameExist = SalesChannel::where('code', '=', $code)->where('name', '=', $name)->where('id', '!=', $id)->where('status', '=', 0)->get();
                if (count($codeAndNameExist) >= 1) {
                    $dataMessage['code'][] = "El codigo y nombre ingresado ya existe";
                    return $this->errorResponse($dataMessage, 422);
                }

                //DATA DEL REQUEST
                $data = $request->except('business_information');
                $data['id'] = $id;
                //ENVIAMOS AL DTO
                $entry = new SalesChannelUpdateDto($data);
                //ACTUALIZAMOS INFORMACION
                $this->saleChannelRepository->update($entry);
                //Nuevamente se consulta la data del ID
                $result = $this->saleChannelRepository->find($id);

                return $this->successResponse($result, "El canal de venta fue actualizado correctamente", 200);
            } else {
                return $this->errorResponse("canal de venta ingresado no existe", 404);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 422);
        }
    }

    public function destroy(int $id)
    {
        if ($this->errorConnection != '') {
            return $this->messageError($this->errorConnection, 500);
        }

        $result = $this->saleChannelRepository->find($id);

        if ($result) {
            $this->saleChannelRepository->destroy($id);

            return $this->successResponse($id, "Canal eliminado correctamente", 200);
        } else {
            return $this->errorResponse("Canal ingresado no existe", 404);
        }
    }
}
